/**
    \file Chamfer2D_3x3.h
    Header file for Chamfer2D_3x3 distance transform class which, given and
    input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
#ifndef Chamfer2D_3x3_h
#define Chamfer2D_3x3_h

#include "DistanceTransform.h"
//----------------------------------------------------------------------
/// Borgefors' Chamfer distance using a 3x3 window
class Chamfer2D_3x3 : public DistanceTransform {

public:
    Chamfer2D_3x3 ( const int xSize, const int ySize, const bool unload=true )
        : DistanceTransform(xSize, ySize, unload)
    {
    }

    /**
        This method simply calls the other doTransform method with the
        following arguments: doTransform(I, 3, 3, 4, true).  Arguments
        of (2,3) have been used to replace (3,4) as well.
     */
    void doTransform ( const unsigned char* const I ) {
        doTransform(I, 3, 3, 4, true);  //or (2,3)
    }

    /**
        This method performs the distance transform.  I is the 2D binary
        input image.  dx & dy are the distances in the x and y directions
        (they need not be the same).  dxy is the distance in the diagonal
        direction.  If halfIsZero is true, then border elements (elements
        of the immediate interior and exterior are initialized with values
        of 0.  If halfIsZero is false, then border elements are initialized
        with dx/2 if the border is in the x direction, dy/2 if the border
        is in the y direction, or min(dx/2, dy/2) for both.
     */
    void doTransform ( const unsigned char* const I, const int dx,
                       const int dy, const int dxy,
                       const bool halfIsZero );

    /**
        This method simply calls the other iterateTransform method with the
        following arguments: iterateTransform(I, 3, 3, 4, true).  Arguments
        of (2,3) have been used to replace (3,4) as well.
     */
    int iterateTransform ( const unsigned char* const I ) {
        return iterateTransform(I, 3, 3, 4, true);  //or (2,3)
    }

    /**
        This method is similar to the doTransform method except that it repeats
        the forward and backward passes of the distance transform algorithm until
        no more distance updates occur (until convergence, in effect).
     */
    int iterateTransform ( const unsigned char* const I, const int dx,
                       const int dy, const int dxy,
                       const bool halfIsZero );

    template <class T>
        inline bool check2 ( T* d, const int center, const int X, const int Y,
                             const T Delta ) {
            const int Near     = sub(X,Y);
            const T   possible = d[Near] + Delta;
            if (possible < d[center]) {
                d[center] = possible;
                return true;
            }
            return false;  //nothing changed
    }

};

#endif
//----------------------------------------------------------------------
